/*
 * @(#)CERConnection.java  1.0  2006-03-20
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */
package ch.hslu.cm.cer.model;

import ch.hslu.cm.simulation.*;
import java.io.IOException;
import java.util.*;
import static ch.hslu.cm.cer.model.ConceptualERModel.*;
import org.jhotdraw.util.ResourceBundleUtil;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;

/**
 * Connects two elements in a Conceptual Entity-Relationship Model.
 * <p>
 * <ul>
 * <li>Represents a relationship when connecting an EntitySet with a
 * RelationshipSet.</li>
 * <li>Represents a generalization or a specialization when connecting an
 * EntitySet with an ISA.</li>
 * <li>Represents an attribute link when connecting an Attribute with an
 * Attribute,  with an EntitySet or with a RelationshipSet.</li>
 * </ul>
 * 
 * 
 * @author Werner Randelshofer
 * @version 1.0 2006-03-20 Created.
 */
public class CERConnection extends AbstractRelationship {

    public final static int MANY_CARDINALITY = -1;
    /*
     * 0 and negative values have special meanings.
     * 1..9999 = count
     */
    private int cardinality = MANY_CARDINALITY;
    private boolean isTraversable = false;
    private boolean isGeneralization;
    private boolean isTotal;

    /** Creates a new instance. */
    public CERConnection() {
    }

    public CERConnection(int cardinality) {
        setCardinality(cardinality);
    }

    public void setCardinality(int newValue) {
        int oldValue = cardinality;
        cardinality = newValue;
        firePropertyChange("cardinality", oldValue, newValue);
    }

    public int getCardinality() {
        return cardinality;
    }

    public void setTraversable(boolean newValue) {
        boolean oldValue = isTraversable;
        isTraversable = newValue;
        firePropertyChange("traversable", oldValue, newValue);
    }

    public boolean isTraversable() {
        return isTraversable;
    }

    public void setGeneralization(boolean newValue) {
        boolean oldValue = isGeneralization;
        isGeneralization = newValue;
        firePropertyChange("generalization", oldValue, newValue);
    }

    public boolean isGeneralization() {
        return isGeneralization;
    }

    public void setTotal(boolean newValue) {
        boolean oldValue = isTotal;
        isTotal = newValue;
        firePropertyChange("total", oldValue, newValue);
    }

    public boolean isTotal() {
        return isTotal;
    }

    @Override
    public int getSimulatedConcept() {
        int sc = getStart().getSimulatedConcept();
        int ec = getEnd().getSimulatedConcept();

        // Represents a relationship when connecting an EntitySet with a
        // RelationshipSet.
        if (sc == ENTITY_SET && ec == RELATIONSHIP_SET ||
                ec == ENTITY_SET && sc == RELATIONSHIP_SET) {

            return RELATIONSHIP;
        }


        // Represents a generalization or a specialization when connecting an
        // EntitySet with an ISA.
        if (sc == ENTITY_SET && ec == ISA ||
                ec == ENTITY_SET && sc == ISA) {

            return (isGeneralization()) ? GENERALIZATION : SPECIALIZATION;
        }

        // Represents an attribute link when connecting an Attribute with an
        // Attribute,  with an EntitySet or with a RelationshipSet.
        if (sc == ATTRIBUTE &&
                (ec == ATTRIBUTE || ec == ENTITY_SET || ec == RELATIONSHIP_SET) ||
                ec == ATTRIBUTE &&
                (sc == ATTRIBUTE || sc == ENTITY_SET || sc == RELATIONSHIP_SET)) {

            return ATTRIBUTE_LINK;
        }

        return UNDEFINED;
    }

    @Override
    public boolean canConnect(SimulatedElement start, SimulatedElement end) {
        int sc = start.getSimulatedConcept();
        int ec = end.getSimulatedConcept();

        // Can connect an EntitySet with a RelationshipSet.
        if (sc == ENTITY_SET && ec == RELATIONSHIP_SET ||
                ec == ENTITY_SET && sc == RELATIONSHIP_SET) {

            return true;
        }


        // Can connect an EntitySet with an ISA.
        if (sc == ENTITY_SET && ec == ISA ||
                ec == ENTITY_SET && sc == ISA) {

            return true;
        }

        // Can connect an attribute with an Attribute, with an EntitySet or with
        // a  RelationshipSet.
        if (sc == ATTRIBUTE &&
                (ec == ATTRIBUTE || ec == ENTITY_SET || ec == RELATIONSHIP_SET) ||
                ec == ATTRIBUTE &&
                (sc == ATTRIBUTE || sc == ENTITY_SET || sc == RELATIONSHIP_SET)) {

            return true;
        }

        return false;
    }

    @Override
    public boolean canConnect(SimulatedElement start) {
        switch (start.getSimulatedConcept()) {
            case ConceptualERModel.ENTITY_SET:
            case ConceptualERModel.RELATIONSHIP_SET:
            case ConceptualERModel.ISA:
            case ConceptualERModel.ATTRIBUTE:
                return true;
            default:
                return false;
        }
    }

    @Override
    public CERConnection clone() {
        CERConnection that = (CERConnection) super.clone();
        return that;
    }

    @Override
    public void read(DOMInput in) throws IOException {
        ResourceBundleUtil labels = ResourceBundleUtil.getBundle("ch.hslu.cm.cer.Labels", Locale.getDefault());

        name = in.getAttribute("name", ResourceBundleUtil.getBundle("ch.hslu.cm.cer.Labels", Locale.getDefault()).getString("relationshipRoleDefaultName"));
        isGeneralization = in.getAttribute("generalization", false);
        isTraversable = in.getAttribute("traversable", false);
        isTotal = in.getAttribute("total", false);

        String value = in.getAttribute("cardinality", "m");
        if (value.equals("m")) {
            cardinality = MANY_CARDINALITY;
        } else {
            cardinality = 1;
        }

        in.openElement((in.getElementCount("start") == 1) ? "start" : "Start");
        in.openElement((in.getElementCount("element") == 1) ? "element" : "Element");
        setStart((SimulatedElement) in.readObject());
        in.closeElement();
        in.closeElement();

        in.openElement((in.getElementCount("end") == 1) ? "end" : "End");
        in.openElement((in.getElementCount("element") == 1) ? "element" : "Element");
        setEnd((SimulatedElement) in.readObject());
        in.closeElement();
        in.closeElement();
    }

    @Override
    public void write(DOMOutput out) throws IOException {
        out.addAttribute("generalization", isGeneralization, false);

        out.addAttribute("name", name, ResourceBundleUtil.getBundle("ch.hslu.cm.cer.Labels", Locale.getDefault()).getString("relationshipRoleDefaultName"));
        out.addAttribute("generalization", isGeneralization, false);
        out.addAttribute("traversable", isTraversable, false);
        out.addAttribute("total", isTotal, false);

        switch (cardinality) {
            case MANY_CARDINALITY:
                out.addAttribute("cardinality", "m");
                break;
            default:
                out.addAttribute("cardinality", cardinality);
        }

        // Write start
        out.openElement("Start");
        out.openElement("Element");
        out.writeObject(getStart());
        out.closeElement();
        out.closeElement();

        // Write end
        out.openElement("End");
        out.openElement("Element");
        out.writeObject(getEnd());
        out.closeElement();
        out.closeElement();
    }

    /**
     * Handles the disconnection of a connection.
     * Override this method to handle this event.
     */
    @Override
    protected void handleDisconnecting(SimulatedElement start, SimulatedElement end) {
    // EntitySet s = (EntitySet) start;
    // EntitySet e = (EntitySet) end;
    }

    /**
     * Handles the connection of a connection.
     * Override this method to handle this event.
     */
    @Override
    protected void handleConnecting(SimulatedElement start, SimulatedElement end) {
    // EntitySet s = (EntitySet) start;
    // EntitySet e = (EntitySet) end;
    }

    @Override
    public boolean isEquivalent(SimulatedObject o) {
        boolean isEquivalent = false;
        if (super.isEquivalent(o) && (o instanceof CERConnection)) {
            CERConnection that = (CERConnection) o;

            switch (getSimulatedConcept()) {
                case RELATIONSHIP:
                    isEquivalent = that.getCardinality() == this.getCardinality() &&
                            that.isTotal() == this.isTotal() &&
                            that.isTraversable() == that.isTraversable();
                    break;
                case GENERALIZATION:
                    isEquivalent = that.isTotal() == this.isTotal() &&
                            that.isTraversable() == that.isTraversable();
                    break;
                case SPECIALIZATION:
                    isEquivalent = that.isTotal() == this.isTotal() &&
                            that.isTraversable() == that.isTraversable();
                    break;
                case ATTRIBUTE_LINK:
                    isEquivalent = true;
                    break;
            }
        }
        return isEquivalent;
    }
}
